import os
import time
import shutil
import subprocess

from config import load_config
from db import DB

QUALITIES = [
    ("360p", 360, 26),
    ("480p", 480, 25),
    ("720p", 720, 24),
]

def run_ffmpeg(in_path: str, out_path: str, height: int, crf: int) -> None:
    # scale keeps aspect ratio; width auto even
    cmd = [
        "ffmpeg", "-y",
        "-i", in_path,
        "-vf", f"scale=-2:{height}",
        "-c:v", "libx264", "-preset", "veryfast", "-crf", str(crf),
        "-c:a", "aac", "-b:a", "128k",
        "-movflags", "+faststart",
        out_path
    ]
    subprocess.run(cmd, check=True, stdout=subprocess.DEVNULL, stderr=subprocess.STDOUT)

def main_once():
    cfg = load_config()
    db = DB(os.path.join(cfg.data_dir, "db.sqlite3"))

    if not shutil.which("ffmpeg"):
        # Mark pending items as error (or keep pending)
        up = db.claim_next_upload()
        if up:
            db.set_upload_status(int(up["id"]), "error", "ffmpeg not found on server")
            db.set_lesson_status(int(up["lesson_id"]), "error")
        return

    up = db.claim_next_upload()
    if not up:
        return

    upload_id = int(up["id"])
    lesson_id = int(up["lesson_id"])
    original_path = up["original_path"]

    try:
        media_dir = os.path.join(cfg.data_dir, "media", str(lesson_id))
        os.makedirs(media_dir, exist_ok=True)

        for label, h, crf in QUALITIES:
            out_path = os.path.join(media_dir, f"{label}.mp4")
            run_ffmpeg(original_path, out_path, h, crf)
            size = os.path.getsize(out_path)
            db.add_media(lesson_id, label, out_path, size)

        db.set_lesson_status(lesson_id, "ready")
        db.set_upload_status(upload_id, "done", None)
    except Exception as e:
        db.set_upload_status(upload_id, "error", str(e))
        db.set_lesson_status(lesson_id, "error")

def main_loop():
    while True:
        main_once()
        time.sleep(3)

if __name__ == "__main__":
    # برای کران بهتره main_once صدا زده بشه
    main_once()